<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class UserCredits extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'current_balance',
        'total_purchased',
        'total_consumed',
    ];

    protected $casts = [
        'current_balance' => 'decimal:2',
        'total_purchased' => 'decimal:2',
        'total_consumed' => 'decimal:2',
    ];

    /**
     * Get the user that owns the credits.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the credit transactions for this user.
     */
    public function transactions()
    {
        return $this->hasMany(CreditTransaction::class, 'user_id', 'user_id');
    }

    /**
     * Add credits to the user's balance.
     */
    public function addCredits($amount)
    {
        $this->current_balance += $amount;
        $this->total_purchased += $amount;
        $this->save();
        
        return $this->current_balance;
    }

    /**
     * Consume credits from the user's balance.
     */
    public function consumeCredits($amount)
    {
        if ($this->current_balance < $amount) {
            throw new \Exception('Insufficient credits');
        }
        
        $this->current_balance -= $amount;
        $this->total_consumed += $amount;
        $this->save();
        
        return $this->current_balance;
    }

    /**
     * Check if user has sufficient credits.
     */
    public function hasSufficientCredits($amount)
    {
        return $this->current_balance >= $amount;
    }

    /**
     * Get the formatted current balance.
     */
    public function getFormattedBalanceAttribute()
    {
        return number_format($this->current_balance, 2);
    }

    /**
     * Get the formatted total purchased.
     */
    public function getFormattedPurchasedAttribute()
    {
        return number_format($this->total_purchased, 2);
    }

    /**
     * Get the formatted total consumed.
     */
    public function getFormattedConsumedAttribute()
    {
        return number_format($this->total_consumed, 2);
    }
}
