<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Tymon\JWTAuth\Contracts\JWTSubject;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable implements JWTSubject
{
    use HasApiTokens, HasFactory, Notifiable, HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'email',
        'password',
        'company_name',
        'company_address',
        'company_ntn',
        'company_phone',
        'province',
        'admin_verified',
        'verified_at',
        'verified_by',
        'session_version',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'admin_verified' => 'boolean',
        'verified_at' => 'datetime',
        'session_version' => 'integer',
    ];

    /**
     * Get the identifier that will be stored in the subject claim of the JWT.
     *
     * @return mixed
     */
    public function getJWTIdentifier()
    {
        return $this->getKey();
    }

    /**
     * Return a key value array, containing any custom claims to be added to the JWT.
     *
     * @return array
     */
    public function getJWTCustomClaims()
    {
        return [];
    }

    /**
     * Get the user's role.
     */
    public function getRoleAttribute()
    {
        return $this->roles->first()?->name ?? 'user';
    }

    /**
     * Check if user is admin.
     */
    public function isAdmin()
    {
        return $this->hasRole('admin');
    }

    /**
     * Check if user is SaaS admin.
     */
    public function isSaasAdmin()
    {
        return $this->hasRole('saas_admin');
    }

    /**
     * Check if user is verified by admin.
     */
    public function isVerified()
    {
        return $this->admin_verified;
    }

    /**
     * Relationships
     */
    public function suppliers()
    {
        return $this->hasMany(Supplier::class);
    }

    public function invoices()
    {
        return $this->hasMany(Invoice::class);
    }

    public function userCredits()
    {
        return $this->hasOne(UserCredits::class);
    }

    public function businesses()
    {
        return $this->hasMany(Business::class);
    }

    public function userSubscriptions()
    {
        return $this->hasMany(UserSubscription::class);
    }

    public function bulkUploads()
    {
        return $this->hasMany(BulkInvoiceUpload::class);
    }

    /**
     * Boot method to set default role
     */
    protected static function boot()
    {
        parent::boot();

        static::created(function ($user) {
            // Assign default 'user' role if no role is assigned
            if (!$user->hasRole('user')) {
                $user->assignRole('user');
            }
        });
    }
}
