<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Supplier extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'email',
        'phone',
        'address',
        'tax_number',
        'shop_cc_no',
        'province',
        'commission_percentage',
        'user_id',
    ];

    protected $casts = [
        'commission_percentage' => 'decimal:2',
    ];

    /**
     * Get the user that owns the supplier.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the invoices for the supplier.
     */
    public function invoices()
    {
        return $this->hasMany(Invoice::class);
    }

    /**
     * Get the province tax rate for the supplier.
     */
    public function provinceTaxRate()
    {
        return $this->belongsTo(ProvinceTaxRate::class, 'province', 'province_name');
    }

    /**
     * Scope a query to only include suppliers for a specific user.
     */
    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope a query to only include suppliers in a specific province.
     */
    public function scopeByProvince($query, $province)
    {
        return $query->where('province', $province);
    }

    /**
     * Scope a query to search suppliers by name or tax number.
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('tax_number', 'like', "%{$search}%")
              ->orWhere('shop_cc_no', 'like', "%{$search}%");
        });
    }

    /**
     * Get the formatted commission percentage.
     */
    public function getFormattedCommissionAttribute()
    {
        return number_format($this->commission_percentage, 2) . '%';
    }

    /**
     * Get the total invoices amount for this supplier.
     */
    public function getTotalInvoicesAmountAttribute()
    {
        return $this->invoices()->sum('total_amount');
    }

    /**
     * Get the total invoices count for this supplier.
     */
    public function getTotalInvoicesCountAttribute()
    {
        return $this->invoices()->count();
    }
}
