<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SaaSInvoice extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'invoice_number',
        'amount',
        'description',
        'status',
        'due_date',
        'paid_at',
        'payment_method',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'due_date' => 'date',
        'paid_at' => 'datetime',
    ];

    /**
     * Get the user that owns the SaaS invoice.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope a query to only include paid invoices.
     */
    public function scopePaid($query)
    {
        return $query->where('status', 'paid');
    }

    /**
     * Scope a query to only include unpaid invoices.
     */
    public function scopeUnpaid($query)
    {
        return $query->where('status', 'unpaid');
    }

    /**
     * Scope a query to only include overdue invoices.
     */
    public function scopeOverdue($query)
    {
        return $query->where('status', 'unpaid')
                    ->where('due_date', '<', now());
    }

    /**
     * Check if invoice is paid.
     */
    public function isPaid()
    {
        return $this->status === 'paid';
    }

    /**
     * Check if invoice is unpaid.
     */
    public function isUnpaid()
    {
        return $this->status === 'unpaid';
    }

    /**
     * Check if invoice is overdue.
     */
    public function isOverdue()
    {
        return $this->isUnpaid() && $this->due_date < now();
    }

    /**
     * Get the formatted amount.
     */
    public function getFormattedAmountAttribute()
    {
        return '$' . number_format($this->amount, 2);
    }

    /**
     * Get the formatted due date.
     */
    public function getFormattedDueDateAttribute()
    {
        return $this->due_date->format('d/m/Y');
    }

    /**
     * Get the formatted paid date.
     */
    public function getFormattedPaidDateAttribute()
    {
        return $this->paid_at ? $this->paid_at->format('d/m/Y') : null;
    }

    /**
     * Get the days until due.
     */
    public function getDaysUntilDueAttribute()
    {
        if ($this->isPaid()) {
            return 0;
        }
        
        return now()->diffInDays($this->due_date, false);
    }
}
