<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class NTNCreditClaim extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'ntn_number',
        'claim_amount',
        'status',
        'submitted_at',
        'processed_at',
        'notes',
    ];

    protected $casts = [
        'claim_amount' => 'decimal:2',
        'submitted_at' => 'datetime',
        'processed_at' => 'datetime',
    ];

    /**
     * Get the user that owns the NTN credit claim.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope a query to only include pending claims.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope a query to only include approved claims.
     */
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    /**
     * Scope a query to only include rejected claims.
     */
    public function scopeRejected($query)
    {
        return $query->where('status', 'rejected');
    }

    /**
     * Check if claim is pending.
     */
    public function isPending()
    {
        return $this->status === 'pending';
    }

    /**
     * Check if claim is approved.
     */
    public function isApproved()
    {
        return $this->status === 'approved';
    }

    /**
     * Check if claim is rejected.
     */
    public function isRejected()
    {
        return $this->status === 'rejected';
    }

    /**
     * Get the formatted claim amount.
     */
    public function getFormattedClaimAmountAttribute()
    {
        return '$' . number_format($this->claim_amount, 2);
    }

    /**
     * Get the formatted submitted date.
     */
    public function getFormattedSubmittedDateAttribute()
    {
        return $this->submitted_at->format('d/m/Y');
    }

    /**
     * Get the formatted processed date.
     */
    public function getFormattedProcessedDateAttribute()
    {
        return $this->processed_at ? $this->processed_at->format('d/m/Y') : null;
    }
}
