<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Invoice extends Model
{
    use HasFactory;

    protected $fillable = [
        'invoice_number',
        'supplier_id',
        'invoice_date',
        'description',
        'subtotal',
        'tax_amount',
        'total_amount',
        'status',
        'user_id',
        'week_number',
        'month_number',
        'year_number',
    ];

    protected $casts = [
        'invoice_date' => 'date',
        'subtotal' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'week_number' => 'integer',
        'month_number' => 'integer',
        'year_number' => 'integer',
    ];

    /**
     * Get the supplier that owns the invoice.
     */
    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * Get the user that owns the invoice.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the invoice items for the invoice.
     */
    public function invoiceItems()
    {
        return $this->hasMany(InvoiceItem::class);
    }

    /**
     * Scope a query to only include invoices for a specific user.
     */
    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope a query to only include invoices with a specific status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope a query to only include invoices for a specific supplier.
     */
    public function scopeBySupplier($query, $supplierId)
    {
        return $query->where('supplier_id', $supplierId);
    }

    /**
     * Scope a query to only include invoices within a date range.
     */
    public function scopeByDateRange($query, $from, $to)
    {
        return $query->whereBetween('invoice_date', [$from, $to]);
    }

    /**
     * Scope a query to exclude credit notes.
     */
    public function scopeExcludeCreditNotes($query)
    {
        return $query->whereNot('invoice_number', 'like', 'INV-CR-%');
    }

    /**
     * Scope a query to only include credit notes.
     */
    public function scopeOnlyCreditNotes($query)
    {
        return $query->where('invoice_number', 'like', 'INV-CR-%');
    }

    /**
     * Scope a query to only include invoices from a specific week.
     */
    public function scopeByWeek($query, $weekNumber)
    {
        return $query->where('week_number', $weekNumber);
    }

    /**
     * Scope a query to only include invoices from a specific month.
     */
    public function scopeByMonth($query, $monthNumber)
    {
        return $query->where('month_number', $monthNumber);
    }

    /**
     * Scope a query to only include invoices from a specific year.
     */
    public function scopeByYear($query, $yearNumber)
    {
        return $query->where('year_number', $yearNumber);
    }

    /**
     * Check if this invoice is a credit note.
     */
    public function isCreditNote()
    {
        return str_starts_with($this->invoice_number, 'INV-CR-');
    }

    /**
     * Get the formatted total amount.
     */
    public function getFormattedTotalAttribute()
    {
        return number_format($this->total_amount, 2);
    }

    /**
     * Get the formatted invoice date.
     */
    public function getFormattedDateAttribute()
    {
        return $this->invoice_date->format('d/m/Y');
    }

    /**
     * Calculate and update week, month, and year numbers.
     */
    public function updateDateNumbers()
    {
        $this->week_number = $this->invoice_date->weekOfYear;
        $this->month_number = $this->invoice_date->month;
        $this->year_number = $this->invoice_date->year;
        $this->save();
    }
}
