<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CreditPurchaseRequest extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'billing_package_id',
        'requested_credits',
        'total_amount',
        'status',
        'submitted_at',
        'processed_at',
        'notes',
    ];

    protected $casts = [
        'requested_credits' => 'integer',
        'total_amount' => 'decimal:2',
        'submitted_at' => 'datetime',
        'processed_at' => 'datetime',
    ];

    /**
     * Get the user that owns the credit purchase request.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the billing package for this request.
     */
    public function billingPackage()
    {
        return $this->belongsTo(BillingPackage::class);
    }

    /**
     * Scope a query to only include pending requests.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope a query to only include approved requests.
     */
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    /**
     * Scope a query to only include rejected requests.
     */
    public function scopeRejected($query)
    {
        return $query->where('status', 'rejected');
    }

    /**
     * Check if request is pending.
     */
    public function isPending()
    {
        return $this->status === 'pending';
    }

    /**
     * Check if request is approved.
     */
    public function isApproved()
    {
        return $this->status === 'approved';
    }

    /**
     * Check if request is rejected.
     */
    public function isRejected()
    {
        return $this->status === 'rejected';
    }

    /**
     * Get the formatted total amount.
     */
    public function getFormattedTotalAmountAttribute()
    {
        return '$' . number_format($this->total_amount, 2);
    }

    /**
     * Get the formatted submitted date.
     */
    public function getFormattedSubmittedDateAttribute()
    {
        return $this->submitted_at->format('d/m/Y');
    }

    /**
     * Get the formatted processed date.
     */
    public function getFormattedProcessedDateAttribute()
    {
        return $this->processed_at ? $this->processed_at->format('d/m/Y') : null;
    }
}
