<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Business extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'business_name',
        'business_address',
        'business_ntn',
        'business_phone',
        'province',
        'status',
        'approved_at',
        'approved_by',
    ];

    protected $casts = [
        'approved_at' => 'datetime',
    ];

    /**
     * Get the user that owns the business.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the business subscriptions for this business.
     */
    public function businessSubscriptions()
    {
        return $this->hasMany(BusinessSubscription::class);
    }

    /**
     * Scope a query to only include approved businesses.
     */
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    /**
     * Scope a query to only include pending businesses.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Check if business is approved.
     */
    public function isApproved()
    {
        return $this->status === 'approved';
    }

    /**
     * Check if business is pending.
     */
    public function isPending()
    {
        return $this->status === 'pending';
    }

    /**
     * Get the formatted approved date.
     */
    public function getFormattedApprovedDateAttribute()
    {
        return $this->approved_at ? $this->approved_at->format('d/m/Y') : null;
    }
}
