<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class BulkInvoiceUpload extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'filename',
        'total_records',
        'processed_records',
        'failed_records',
        'status',
        'error_log',
    ];

    protected $casts = [
        'total_records' => 'integer',
        'processed_records' => 'integer',
        'failed_records' => 'integer',
        'error_log' => 'array',
    ];

    /**
     * Get the user that owns the bulk upload.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope a query to only include completed uploads.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope a query to only include failed uploads.
     */
    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    /**
     * Scope a query to only include processing uploads.
     */
    public function scopeProcessing($query)
    {
        return $query->where('status', 'processing');
    }

    /**
     * Check if upload is completed.
     */
    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    /**
     * Check if upload is failed.
     */
    public function isFailed()
    {
        return $this->status === 'failed';
    }

    /**
     * Check if upload is processing.
     */
    public function isProcessing()
    {
        return $this->status === 'processing';
    }

    /**
     * Get the success rate percentage.
     */
    public function getSuccessRateAttribute()
    {
        if ($this->total_records === 0) {
            return 0;
        }
        
        return round(($this->processed_records / $this->total_records) * 100, 2);
    }

    /**
     * Get the formatted success rate.
     */
    public function getFormattedSuccessRateAttribute()
    {
        return $this->success_rate . '%';
    }
}
