<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AdminSettings extends Model
{
    use HasFactory;

    protected $fillable = [
        'key',
        'value',
        'description',
        'type',
        'is_public',
    ];

    protected $casts = [
        'is_public' => 'boolean',
    ];

    /**
     * Get setting value by key.
     */
    public static function getValue($key, $default = null)
    {
        $setting = static::where('key', $key)->first();
        return $setting ? $setting->value : $default;
    }

    /**
     * Set setting value by key.
     */
    public static function setValue($key, $value, $description = null, $type = 'string', $isPublic = false)
    {
        $setting = static::updateOrCreate(
            ['key' => $key],
            [
                'value' => $value,
                'description' => $description,
                'type' => $type,
                'is_public' => $isPublic,
            ]
        );
        
        return $setting;
    }

    /**
     * Scope a query to only include public settings.
     */
    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    /**
     * Scope a query to only include private settings.
     */
    public function scopePrivate($query)
    {
        return $query->where('is_public', false);
    }

    /**
     * Get the formatted value based on type.
     */
    public function getFormattedValueAttribute()
    {
        switch ($this->type) {
            case 'boolean':
                return $this->value ? 'Yes' : 'No';
            case 'number':
                return number_format($this->value);
            case 'currency':
                return '$' . number_format($this->value, 2);
            case 'percentage':
                return number_format($this->value, 2) . '%';
            default:
                return $this->value;
        }
    }

    /**
     * Check if setting is public.
     */
    public function isPublic()
    {
        return $this->is_public;
    }

    /**
     * Check if setting is private.
     */
    public function isPrivate()
    {
        return !$this->is_public;
    }
}
