<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\UserCredits;
use App\Models\CreditTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class UserCreditsController extends Controller
{
    /**
     * Display user credits information.
     */
    public function index()
    {
        $user = Auth::user();
        
        $userCredits = UserCredits::where('user_id', $user->id)->first();
        
        if (!$userCredits) {
            // Create user credits record if it doesn't exist
            $userCredits = UserCredits::create([
                'user_id' => $user->id,
                'current_balance' => 0,
                'total_purchased' => 0,
                'total_consumed' => 0,
            ]);
        }
        
        // Get recent transactions
        $recentTransactions = CreditTransaction::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
        
        return response()->json([
            'success' => true,
            'data' => [
                'credits' => [
                    'current_balance' => $userCredits->current_balance,
                    'total_purchased' => $userCredits->total_purchased,
                    'total_consumed' => $userCredits->total_consumed,
                    'formatted_balance' => $userCredits->formatted_balance,
                    'formatted_purchased' => $userCredits->formatted_purchased,
                    'formatted_consumed' => $userCredits->formatted_consumed,
                ],
                'recent_transactions' => $recentTransactions,
            ]
        ]);
    }

    /**
     * Consume credits for an operation.
     */
    public function consume(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'description' => 'nullable|string|max:255',
            'reference_id' => 'nullable|integer',
            'reference_type' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        $amount = $request->amount;
        
        $userCredits = UserCredits::where('user_id', $user->id)->first();
        
        if (!$userCredits) {
            return response()->json([
                'success' => false,
                'message' => 'User credits record not found'
            ], 404);
        }
        
        // Check if user has sufficient credits
        if (!$userCredits->hasSufficientCredits($amount)) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient credits',
                'data' => [
                    'required' => $amount,
                    'available' => $userCredits->current_balance,
                ]
            ], 400);
        }
        
        try {
            // Consume credits
            $newBalance = $userCredits->consumeCredits($amount);
            
            // Create transaction record
            CreditTransaction::create([
                'user_id' => $user->id,
                'amount' => $amount,
                'type' => 'debit',
                'description' => $request->description ?? 'Credit consumption',
                'reference_id' => $request->reference_id,
                'reference_type' => $request->reference_type,
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Credits consumed successfully',
                'data' => [
                    'amount_consumed' => $amount,
                    'new_balance' => $newBalance,
                    'formatted_new_balance' => number_format($newBalance, 2),
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error consuming credits: ' . $e->getMessage()
            ], 500);
        }
    }
}
