<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class SupplierController extends Controller
{
    /**
     * Display a listing of suppliers.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        $query = Supplier::forUser($user->id);
        
        // Search functionality
        if ($request->has('search')) {
            $query->search($request->search);
        }
        
        // Filter by province
        if ($request->has('province')) {
            $query->byProvince($request->province);
        }
        
        $suppliers = $query->orderBy('name')->get();
        
        return response()->json([
            'success' => true,
            'data' => $suppliers
        ]);
    }

    /**
     * Store a newly created supplier.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:500',
            'tax_number' => 'nullable|string|max:255',
            'shop_cc_no' => 'nullable|string|max:255',
            'province' => 'required|string|max:255',
            'commission_percentage' => 'nullable|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        
        $supplier = Supplier::create([
            'user_id' => $user->id,
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'address' => $request->address,
            'tax_number' => $request->tax_number,
            'shop_cc_no' => $request->shop_cc_no,
            'province' => $request->province,
            'commission_percentage' => $request->commission_percentage ?? 0,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Supplier created successfully',
            'data' => $supplier
        ], 201);
    }

    /**
     * Display the specified supplier.
     */
    public function show($id)
    {
        $user = Auth::user();
        
        $supplier = Supplier::forUser($user->id)->findOrFail($id);
        
        return response()->json([
            'success' => true,
            'data' => $supplier
        ]);
    }

    /**
     * Update the specified supplier.
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:500',
            'tax_number' => 'nullable|string|max:255',
            'shop_cc_no' => 'nullable|string|max:255',
            'province' => 'required|string|max:255',
            'commission_percentage' => 'nullable|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        
        $supplier = Supplier::forUser($user->id)->findOrFail($id);
        
        $supplier->update([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'address' => $request->address,
            'tax_number' => $request->tax_number,
            'shop_cc_no' => $request->shop_cc_no,
            'province' => $request->province,
            'commission_percentage' => $request->commission_percentage ?? 0,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Supplier updated successfully',
            'data' => $supplier
        ]);
    }

    /**
     * Remove the specified supplier.
     */
    public function destroy($id)
    {
        $user = Auth::user();
        
        $supplier = Supplier::forUser($user->id)->findOrFail($id);
        
        // Check if supplier has invoices
        if ($supplier->invoices()->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete supplier with existing invoices'
            ], 400);
        }
        
        $supplier->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'Supplier deleted successfully'
        ]);
    }
}
