<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Supplier;
use App\Models\ProvinceTaxRate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class InvoiceController extends Controller
{
    /**
     * Display a listing of invoices.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        $query = Invoice::forUser($user->id)->with(['supplier', 'invoiceItems']);
        
        // Filter by supplier
        if ($request->has('supplier_id')) {
            $query->bySupplier($request->supplier_id);
        }
        
        // Filter by status
        if ($request->has('status')) {
            $query->byStatus($request->status);
        }
        
        // Filter by date range
        if ($request->has('from_date') && $request->has('to_date')) {
            $query->byDateRange($request->from_date, $request->to_date);
        }
        
        // Filter by week
        if ($request->has('week')) {
            $query->byWeek($request->week);
        }
        
        // Filter by month
        if ($request->has('month')) {
            $query->byMonth($request->month);
        }
        
        // Filter by year
        if ($request->has('year')) {
            $query->byYear($request->year);
        }
        
        $invoices = $query->orderBy('invoice_date', 'desc')->get();
        
        return response()->json([
            'success' => true,
            'data' => $invoices
        ]);
    }

    /**
     * Store a newly created invoice.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'supplier_id' => 'required|exists:suppliers,id',
            'invoice_number' => 'required|string|max:255',
            'invoice_date' => 'required|date',
            'description' => 'nullable|string|max:500',
            'items' => 'required|array|min:1',
            'items.*.item_name' => 'required|string|max:255',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        
        // Verify supplier belongs to user
        $supplier = Supplier::forUser($user->id)->findOrFail($request->supplier_id);
        
        DB::beginTransaction();
        
        try {
            // Calculate totals
            $subtotal = 0;
            $taxAmount = 0;
            
            foreach ($request->items as $item) {
                $itemTotal = $item['quantity'] * $item['unit_price'];
                $subtotal += $itemTotal;
                
                if (isset($item['tax_rate'])) {
                    $taxAmount += $itemTotal * ($item['tax_rate'] / 100);
                }
            }
            
            $totalAmount = $subtotal + $taxAmount;
            
            // Create invoice
            $invoice = Invoice::create([
                'user_id' => $user->id,
                'supplier_id' => $supplier->id,
                'invoice_number' => $request->invoice_number,
                'invoice_date' => $request->invoice_date,
                'description' => $request->description,
                'subtotal' => $subtotal,
                'tax_amount' => $taxAmount,
                'total_amount' => $totalAmount,
                'status' => 'active',
                'week_number' => date('W', strtotime($request->invoice_date)),
                'month_number' => date('n', strtotime($request->invoice_date)),
                'year_number' => date('Y', strtotime($request->invoice_date)),
            ]);
            
            // Create invoice items
            foreach ($request->items as $item) {
                InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'item_name' => $item['item_name'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'tax_rate' => $item['tax_rate'] ?? 0,
                    'total_amount' => $item['quantity'] * $item['unit_price'],
                ]);
            }
            
            DB::commit();
            
            $invoice->load(['supplier', 'invoiceItems']);
            
            return response()->json([
                'success' => true,
                'message' => 'Invoice created successfully',
                'data' => $invoice
            ], 201);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Error creating invoice: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified invoice.
     */
    public function show($id)
    {
        $user = Auth::user();
        
        $invoice = Invoice::forUser($user->id)
            ->with(['supplier', 'invoiceItems'])
            ->findOrFail($id);
        
        return response()->json([
            'success' => true,
            'data' => $invoice
        ]);
    }

    /**
     * Update the specified invoice.
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'supplier_id' => 'required|exists:suppliers,id',
            'invoice_number' => 'required|string|max:255',
            'invoice_date' => 'required|date',
            'description' => 'nullable|string|max:500',
            'items' => 'required|array|min:1',
            'items.*.item_name' => 'required|string|max:255',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        
        // Verify invoice belongs to user
        $invoice = Invoice::forUser($user->id)->findOrFail($id);
        
        // Verify supplier belongs to user
        $supplier = Supplier::forUser($user->id)->findOrFail($request->supplier_id);
        
        DB::beginTransaction();
        
        try {
            // Calculate totals
            $subtotal = 0;
            $taxAmount = 0;
            
            foreach ($request->items as $item) {
                $itemTotal = $item['quantity'] * $item['unit_price'];
                $subtotal += $itemTotal;
                
                if (isset($item['tax_rate'])) {
                    $taxAmount += $itemTotal * ($item['tax_rate'] / 100);
                }
            }
            
            $totalAmount = $subtotal + $taxAmount;
            
            // Update invoice
            $invoice->update([
                'supplier_id' => $supplier->id,
                'invoice_number' => $request->invoice_number,
                'invoice_date' => $request->invoice_date,
                'description' => $request->description,
                'subtotal' => $subtotal,
                'tax_amount' => $taxAmount,
                'total_amount' => $totalAmount,
                'week_number' => date('W', strtotime($request->invoice_date)),
                'month_number' => date('n', strtotime($request->invoice_date)),
                'year_number' => date('Y', strtotime($request->invoice_date)),
            ]);
            
            // Delete existing items
            $invoice->invoiceItems()->delete();
            
            // Create new invoice items
            foreach ($request->items as $item) {
                InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'item_name' => $item['item_name'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'tax_rate' => $item['tax_rate'] ?? 0,
                    'total_amount' => $item['quantity'] * $item['unit_price'],
                ]);
            }
            
            DB::commit();
            
            $invoice->load(['supplier', 'invoiceItems']);
            
            return response()->json([
                'success' => true,
                'message' => 'Invoice updated successfully',
                'data' => $invoice
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating invoice: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified invoice.
     */
    public function destroy($id)
    {
        $user = Auth::user();
        
        $invoice = Invoice::forUser($user->id)->findOrFail($id);
        
        DB::beginTransaction();
        
        try {
            // Delete invoice items first
            $invoice->invoiceItems()->delete();
            
            // Delete invoice
            $invoice->delete();
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Invoice deleted successfully'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Error deleting invoice: ' . $e->getMessage()
            ], 500);
        }
    }
}
