<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Supplier;
use App\Models\UserCredits;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Display dashboard data.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        // Get current week, month, and year
        $currentWeek = date('W');
        $currentMonth = date('n');
        $currentYear = date('Y');
        
        // Get user's credits
        $userCredits = UserCredits::where('user_id', $user->id)->first();
        $currentCredits = $userCredits ? $userCredits->current_balance : 0;
        
        // Get invoice statistics
        $totalInvoices = Invoice::forUser($user->id)->count();
        $totalAmount = Invoice::forUser($user->id)->sum('total_amount');
        
        // Get current week invoices
        $currentWeekInvoices = Invoice::forUser($user->id)
            ->byWeek($currentWeek)
            ->byYear($currentYear)
            ->count();
        $currentWeekAmount = Invoice::forUser($user->id)
            ->byWeek($currentWeek)
            ->byYear($currentYear)
            ->sum('total_amount');
        
        // Get current month invoices
        $currentMonthInvoices = Invoice::forUser($user->id)
            ->byMonth($currentMonth)
            ->byYear($currentYear)
            ->count();
        $currentMonthAmount = Invoice::forUser($user->id)
            ->byMonth($currentMonth)
            ->byYear($currentYear)
            ->sum('total_amount');
        
        // Get current year invoices
        $currentYearInvoices = Invoice::forUser($user->id)
            ->byYear($currentYear)
            ->count();
        $currentYearAmount = Invoice::forUser($user->id)
            ->byYear($currentYear)
            ->sum('total_amount');
        
        // Get supplier count
        $totalSuppliers = Supplier::forUser($user->id)->count();
        
        // Get recent invoices
        $recentInvoices = Invoice::forUser($user->id)
            ->with('supplier')
            ->orderBy('invoice_date', 'desc')
            ->limit(5)
            ->get();
        
        // Get weekly breakdown for current year
        $weeklyBreakdown = Invoice::forUser($user->id)
            ->byYear($currentYear)
            ->selectRaw('week_number, COUNT(*) as count, SUM(total_amount) as total')
            ->groupBy('week_number')
            ->orderBy('week_number')
            ->get();
        
        // Get monthly breakdown for current year
        $monthlyBreakdown = Invoice::forUser($user->id)
            ->byYear($currentYear)
            ->selectRaw('month_number, COUNT(*) as count, SUM(total_amount) as total')
            ->groupBy('month_number')
            ->orderBy('month_number')
            ->get();
        
        // Get top suppliers by invoice count
        $topSuppliers = Supplier::forUser($user->id)
            ->withCount('invoices')
            ->withSum('invoices', 'total_amount')
            ->orderByDesc('invoices_count')
            ->limit(5)
            ->get();
        
        return response()->json([
            'success' => true,
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'email' => $user->email,
                    'company_name' => $user->company_name,
                    'province' => $user->province,
                    'admin_verified' => $user->admin_verified,
                ],
                'credits' => [
                    'current_balance' => $currentCredits,
                    'formatted_balance' => number_format($currentCredits, 2),
                ],
                'statistics' => [
                    'total_invoices' => $totalInvoices,
                    'total_amount' => $totalAmount,
                    'formatted_total_amount' => number_format($totalAmount, 2),
                    'total_suppliers' => $totalSuppliers,
                ],
                'current_periods' => [
                    'week' => [
                        'number' => $currentWeek,
                        'invoices' => $currentWeekInvoices,
                        'amount' => $currentWeekAmount,
                        'formatted_amount' => number_format($currentWeekAmount, 2),
                    ],
                    'month' => [
                        'number' => $currentMonth,
                        'invoices' => $currentMonthInvoices,
                        'amount' => $currentMonthAmount,
                        'formatted_amount' => number_format($currentMonthAmount, 2),
                    ],
                    'year' => [
                        'number' => $currentYear,
                        'invoices' => $currentYearInvoices,
                        'amount' => $currentYearAmount,
                        'formatted_amount' => number_format($currentYearAmount, 2),
                    ],
                ],
                'recent_invoices' => $recentInvoices,
                'weekly_breakdown' => $weeklyBreakdown,
                'monthly_breakdown' => $monthlyBreakdown,
                'top_suppliers' => $topSuppliers,
            ]
        ]);
    }
}
