<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Business;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class BusinessController extends Controller
{
    /**
     * Display a listing of businesses.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        $query = Business::where('user_id', $user->id);
        
        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }
        
        $businesses = $query->orderBy('business_name')->get();
        
        return response()->json([
            'success' => true,
            'data' => $businesses
        ]);
    }

    /**
     * Store a newly created business.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'business_name' => 'required|string|max:255',
            'business_address' => 'required|string|max:500',
            'business_ntn' => 'required|string|max:255',
            'business_phone' => 'required|string|max:255',
            'province' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        
        $business = Business::create([
            'user_id' => $user->id,
            'business_name' => $request->business_name,
            'business_address' => $request->business_address,
            'business_ntn' => $request->business_ntn,
            'business_phone' => $request->business_phone,
            'province' => $request->province,
            'status' => 'pending',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Business created successfully. Please wait for admin approval.',
            'data' => $business
        ], 201);
    }

    /**
     * Display the specified business.
     */
    public function show($id)
    {
        $user = Auth::user();
        
        $business = Business::where('user_id', $user->id)->findOrFail($id);
        
        return response()->json([
            'success' => true,
            'data' => $business
        ]);
    }

    /**
     * Update the specified business.
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'business_name' => 'required|string|max:255',
            'business_address' => 'required|string|max:500',
            'business_ntn' => 'required|string|max:255',
            'business_phone' => 'required|string|max:255',
            'province' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        
        $business = Business::where('user_id', $user->id)->findOrFail($id);
        
        // Reset status to pending if business details are updated
        $business->update([
            'business_name' => $request->business_name,
            'business_address' => $request->business_address,
            'business_ntn' => $request->business_ntn,
            'business_phone' => $request->business_phone,
            'province' => $request->province,
            'status' => 'pending',
            'approved_at' => null,
            'approved_by' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Business updated successfully. Please wait for admin approval.',
            'data' => $business
        ]);
    }

    /**
     * Remove the specified business.
     */
    public function destroy($id)
    {
        $user = Auth::user();
        
        $business = Business::where('user_id', $user->id)->findOrFail($id);
        
        // Check if business has active subscriptions
        if ($business->businessSubscriptions()->active()->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete business with active subscriptions'
            ], 400);
        }
        
        $business->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'Business deleted successfully'
        ]);
    }
}
