<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\BulkInvoiceUpload;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class BulkUploadController extends Controller
{
    /**
     * Display a listing of bulk uploads.
     */
    public function index()
    {
        $user = Auth::user();
        
        $bulkUploads = BulkInvoiceUpload::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();
        
        return response()->json([
            'success' => true,
            'data' => $bulkUploads
        ]);
    }

    /**
     * Import CSV file.
     */
    public function import(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'csv_file' => 'required|file|mimes:csv,txt|max:10240', // 10MB max
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        
        try {
            $file = $request->file('csv_file');
            $filename = $file->getClientOriginalName();
            
            // Store file
            $path = $file->store('bulk-uploads', 'public');
            
            // Create bulk upload record
            $bulkUpload = BulkInvoiceUpload::create([
                'user_id' => $user->id,
                'filename' => $filename,
                'total_records' => 0, // Will be updated after processing
                'processed_records' => 0,
                'failed_records' => 0,
                'status' => 'processing',
                'error_log' => [],
            ]);
            
            // Here you would implement the actual CSV processing logic
            // For now, just mark as completed
            $bulkUpload->update([
                'status' => 'completed',
                'total_records' => 10, // Example
                'processed_records' => 8, // Example
                'failed_records' => 2, // Example
                'error_log' => [
                    'Row 3: Invalid supplier ID',
                    'Row 7: Missing required field'
                ],
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'CSV file uploaded successfully',
                'data' => $bulkUpload
            ], 201);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error uploading CSV: ' . $e->getMessage()
            ], 500);
        }
    }
}
