<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Facades\JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;

class AuthController extends Controller
{
    /**
     * Create a new AuthController instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth:api', ['except' => ['login', 'register', 'forgotPassword', 'resetPassword']]);
    }

    /**
     * Get a JWT via given credentials.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|string|min:6',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $credentials = $request->only('email', 'password');

        try {
            if (!$token = JWTAuth::attempt($credentials)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid credentials'
                ], 401);
            }

            $user = Auth::user();
            
            // Check if user is verified
            if (!$user->isVerified()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Your account is pending admin verification. Please wait for approval.'
                ], 403);
            }

            return response()->json([
                'success' => true,
                'message' => 'Login successful',
                'access_token' => $token,
                'token_type' => 'bearer',
                'expires_in' => auth('api')->factory()->getTTL() * 60,
                'user' => [
                    'id' => $user->id,
                    'email' => $user->email,
                    'role' => $user->role,
                    'company_name' => $user->company_name,
                    'company_address' => $user->company_address,
                    'company_ntn' => $user->company_ntn,
                    'company_phone' => $user->company_phone,
                    'province' => $user->province,
                    'admin_verified' => $user->admin_verified,
                    'verified_at' => $user->verified_at,
                    'session_version' => $user->session_version,
                ]
            ]);

        } catch (JWTException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Could not create token'
            ], 500);
        }
    }

    /**
     * Register a new user.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:6|confirmed',
            'company_name' => 'required|string|max:255',
            'company_address' => 'required|string|max:500',
            'company_ntn' => 'required|string|max:255',
            'company_phone' => 'required|string|max:255',
            'province' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::create([
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'company_name' => $request->company_name,
            'company_address' => $request->company_address,
            'company_ntn' => $request->company_ntn,
            'company_phone' => $request->company_phone,
            'province' => $request->province,
            'admin_verified' => false,
            'session_version' => 1,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'User registered successfully. Please wait for admin verification.',
            'user' => [
                'id' => $user->id,
                'email' => $user->email,
                'company_name' => $user->company_name,
                'admin_verified' => $user->admin_verified,
            ]
        ], 201);
    }

    /**
     * Get the authenticated User.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function me()
    {
        $user = Auth::user();
        
        return response()->json([
            'success' => true,
            'data' => [
                'id' => $user->id,
                'email' => $user->email,
                'role' => $user->role,
                'company_name' => $user->company_name,
                'company_address' => $user->company_address,
                'company_ntn' => $user->company_ntn,
                'company_phone' => $user->company_phone,
                'province' => $user->province,
                'admin_verified' => $user->admin_verified,
                'verified_at' => $user->verified_at,
                'session_version' => $user->session_version,
            ]
        ]);
    }

    /**
     * Log the user out (Invalidate the token).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout()
    {
        try {
            JWTAuth::invalidate(JWTAuth::getToken());
            
            return response()->json([
                'success' => true,
                'message' => 'Successfully logged out'
            ]);
        } catch (JWTException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error logging out'
            ], 500);
        }
    }

    /**
     * Refresh a token.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function refresh()
    {
        try {
            $token = JWTAuth::refresh(JWTAuth::getToken());
            $user = Auth::user();
            
            return response()->json([
                'success' => true,
                'message' => 'Token refreshed successfully',
                'access_token' => $token,
                'token_type' => 'bearer',
                'expires_in' => auth('api')->factory()->getTTL() * 60,
                'user' => [
                    'id' => $user->id,
                    'email' => $user->email,
                    'role' => $user->role,
                    'company_name' => $user->company_name,
                    'company_address' => $user->company_address,
                    'company_ntn' => $user->company_ntn,
                    'company_phone' => $user->company_phone,
                    'province' => $user->province,
                    'admin_verified' => $user->admin_verified,
                    'verified_at' => $user->verified_at,
                    'session_version' => $user->session_version,
                ]
            ]);
        } catch (JWTException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error refreshing token'
            ], 500);
        }
    }

    /**
     * Verify session version.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function verifySession(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'client_version' => 'required|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        $clientVersion = $request->client_version;
        $serverVersion = $user->session_version;

        if ($clientVersion < $serverVersion) {
            return response()->json([
                'success' => false,
                'message' => 'Session version mismatch',
                'data' => [
                    'server_version' => $serverVersion,
                    'client_version' => $clientVersion
                ]
            ], 409);
        }

        return response()->json([
            'success' => true,
            'message' => 'Session version verified',
            'data' => [
                'server_version' => $serverVersion,
                'client_version' => $clientVersion
            ]
        ]);
    }

    /**
     * Increment session version.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function incrementSession()
    {
        $user = Auth::user();
        $user->increment('session_version');
        $user->refresh();

        return response()->json([
            'success' => true,
            'message' => 'Session version incremented',
            'data' => [
                'new_version' => $user->session_version
            ]
        ]);
    }

    /**
     * Send password reset email.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function forgotPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        // Here you would implement password reset logic
        // For now, just return success message
        
        return response()->json([
            'success' => true,
            'message' => 'Password reset email sent successfully'
        ]);
    }

    /**
     * Reset password with token.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'token' => 'required|string',
            'email' => 'required|email|exists:users,email',
            'password' => 'required|string|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        // Here you would implement password reset logic
        // For now, just return success message
        
        return response()->json([
            'success' => true,
            'message' => 'Password reset successfully'
        ]);
    }

    /**
     * Change password for authenticated user.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function changePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'current_password' => 'required|string',
            'new_password' => 'required|string|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();

        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect'
            ], 400);
        }

        $user->update([
            'password' => Hash::make($request->new_password)
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully'
        ]);
    }
}
