# 🚀 Migration Guide: Supabase to Laravel Backend

This guide will help you migrate your existing React frontend from Supabase to the new Laravel backend while keeping your frontend layout and design intact.

## 📋 **What We're Migrating**

- ✅ **Backend**: Supabase → Laravel 11 + PostgreSQL
- ✅ **Authentication**: Supabase Auth → JWT Tokens
- ✅ **Database**: Supabase → PostgreSQL (same data, different backend)
- 🔄 **Frontend**: Keep existing React components, update API calls only

## 🎯 **Migration Strategy**

1. **Phase 1**: Set up Laravel backend (✅ COMPLETED)
2. **Phase 2**: Update frontend integration files (✅ COMPLETED)
3. **Phase 3**: Test both systems side by side
4. **Phase 4**: Switch endpoints gradually
5. **Phase 5**: Remove Supabase code

## 🔧 **Step 1: Update Your Main App Component**

### **File: `src/App.tsx`**

**BEFORE (Supabase):**
```tsx
import { AuthProvider } from './components/AuthProvider'
```

**AFTER (Laravel):**
```tsx
import { LaravelAuthProvider } from './components/LaravelAuthProvider'
```

**Change the provider wrapper:**
```tsx
// OLD
<AuthProvider>
  {/* Your app content */}
</AuthProvider>

// NEW
<LaravelAuthProvider>
  {/* Your app content */}
</LaravelAuthProvider>
```

## 🔧 **Step 2: Update Authentication Components**

### **File: `src/components/LoginForm.tsx` (if exists)**

**BEFORE (Supabase):**
```tsx
import { useAuth } from '../hooks/useAuth'
// or
import { useAuth } from '../components/AuthProvider'

const { signIn } = useAuth()
```

**AFTER (Laravel):**
```tsx
import { useLaravelAuth } from '../components/LaravelAuthProvider'

const { signIn } = useLaravelAuth()
```

### **File: `src/components/RegisterForm.tsx` (if exists)**

**BEFORE (Supabase):**
```tsx
import { useAuth } from '../hooks/useAuth'
const { signUp } = useAuth()
```

**AFTER (Laravel):**
```tsx
import { useLaravelAuth } from '../components/LaravelAuthProvider'
const { signUp } = useLaravelAuth()
```

## 🔧 **Step 3: Update API Integration Files**

### **File: `src/hooks/useCredits.ts`**

**BEFORE (Supabase):**
```tsx
import { supabase } from '../integrations/supabase/client'

export const useCredits = () => {
  const fetchCredits = async () => {
    const { data, error } = await supabase
      .from('user_credits')
      .select('*')
      .eq('user_id', userId)
      .single()
    // ... rest of the code
  }
}
```

**AFTER (Laravel):**
```tsx
import { laravelClient } from '../integrations/laravel/client'

export const useCredits = () => {
  const fetchCredits = async () => {
    try {
      const response = await laravelClient.getUserCredits()
      if (response.success && response.data) {
        return response.data
      }
    } catch (error) {
      console.error('Error fetching credits:', error)
    }
  }
}
```

### **File: `src/pages/Dashboard.tsx`**

**BEFORE (Supabase):**
```tsx
import { supabase } from '../integrations/supabase/client'

const fetchDashboardData = async () => {
  const { data: invoices } = await supabase
    .from('invoices')
    .select('*')
    .eq('user_id', user.id)
    .limit(5)
}
```

**AFTER (Laravel):**
```tsx
import { laravelClient } from '../integrations/laravel/client'

const fetchDashboardData = async () => {
  try {
    const response = await laravelClient.getDashboard()
    if (response.success && response.data) {
      return response.data
    }
  } catch (error) {
    console.error('Error fetching dashboard data:', error)
  }
}
```

## 🔧 **Step 4: Update Specific Page Components**

### **File: `src/pages/Suppliers.tsx`**

**BEFORE (Supabase):**
```tsx
const fetchSuppliers = async () => {
  const { data, error } = await supabase
    .from('suppliers')
    .select('*')
    .eq('user_id', user.id)
}
```

**AFTER (Laravel):**
```tsx
const fetchSuppliers = async () => {
  try {
    const response = await laravelClient.getSuppliers()
    if (response.success && response.data) {
      setSuppliers(response.data)
    }
  } catch (error) {
    console.error('Error fetching suppliers:', error)
  }
}
```

### **File: `src/pages/AllInvoices.tsx`**

**BEFORE (Supabase):**
```tsx
const fetchInvoices = async () => {
  const { data, error } = await supabase
    .from('invoices')
    .select('*, supplier:suppliers(*)')
    .eq('user_id', user.id)
}
```

**AFTER (Laravel):**
```tsx
const fetchInvoices = async () => {
  try {
    const response = await laravelClient.getInvoices()
    if (response.success && response.data) {
      setInvoices(response.data)
    }
  } catch (error) {
    console.error('Error fetching invoices:', error)
  }
}
```

## 🔧 **Step 5: Update Form Submissions**

### **File: `src/pages/InvoiceCreate.tsx`**

**BEFORE (Supabase):**
```tsx
const handleSubmit = async (data: any) => {
  const { data: invoice, error } = await supabase
    .from('invoices')
    .insert([data])
    .select()
    .single()
}
```

**AFTER (Laravel):**
```tsx
const handleSubmit = async (data: any) => {
  try {
    const response = await laravelClient.createInvoice(data)
    if (response.success) {
      // Handle success
      navigate('/invoices')
    }
  } catch (error) {
    console.error('Error creating invoice:', error)
  }
}
```

## 🔧 **Step 6: Update File Operations**

### **File: `src/pages/BulkUpload.tsx`**

**BEFORE (Supabase):**
```tsx
const handleFileUpload = async (file: File) => {
  const { data, error } = await supabase.storage
    .from('bulk-uploads')
    .upload(`${userId}/${file.name}`, file)
}
```

**AFTER (Laravel):**
```tsx
const handleFileUpload = async (file: File) => {
  try {
    const response = await laravelClient.uploadCSV(file)
    if (response.success) {
      // Handle success
    }
  } catch (error) {
    console.error('Error uploading file:', error)
  }
}
```

## 🔧 **Step 7: Update Environment Variables**

### **File: `.env`**

**BEFORE (Supabase):**
```env
VITE_SUPABASE_URL=your_supabase_url
VITE_SUPABASE_ANON_KEY=your_supabase_key
```

**AFTER (Laravel):**
```env
VITE_LARAVEL_API_URL=http://localhost:8000/api
# For production:
# VITE_LARAVEL_API_URL=https://yourdomain.com/api
```

## 🔧 **Step 8: Update API Base URL**

### **File: `src/integrations/laravel/client.ts`**

**Change the default base URL:**
```tsx
constructor(baseUrl: string = import.meta.env.VITE_LARAVEL_API_URL || 'http://localhost:8000/api') {
  this.baseUrl = baseUrl
  this.token = this.getStoredToken()
}
```

## 🧪 **Testing Strategy**

### **Phase 1: Side-by-Side Testing**
1. Keep both Supabase and Laravel running
2. Test Laravel endpoints with Postman/Insomnia
3. Update one component at a time
4. Test thoroughly before moving to the next

### **Phase 2: Gradual Switch**
1. Start with authentication (login/register)
2. Move to read operations (fetching data)
3. Update create/update/delete operations
4. Test file uploads and downloads

### **Phase 3: Full Migration**
1. Remove Supabase imports
2. Clean up unused code
3. Test the entire application
4. Deploy to production

## 📁 **Files That Need Updates**

### **High Priority (Update First):**
- `src/App.tsx` - Change AuthProvider
- `src/components/LoginForm.tsx` - Update auth hooks
- `src/components/RegisterForm.tsx` - Update auth hooks
- `src/pages/Dashboard.tsx` - Update data fetching

### **Medium Priority:**
- `src/pages/Suppliers.tsx` - Update CRUD operations
- `src/pages/AllInvoices.tsx` - Update data fetching
- `src/pages/InvoiceCreate.tsx` - Update form submission
- `src/hooks/useCredits.ts` - Update API calls

### **Low Priority:**
- `src/pages/BulkUpload.tsx` - Update file operations
- `src/pages/CreditNotes.tsx` - Update API calls
- Other utility components

## 🚨 **Common Issues & Solutions**

### **Issue 1: CORS Errors**
**Solution**: Ensure Laravel CORS is configured properly in `config/cors.php`

### **Issue 2: JWT Token Not Working**
**Solution**: Check if token is being sent in Authorization header

### **Issue 3: API Endpoints Not Found**
**Solution**: Verify Laravel routes are defined in `routes/web.php`

### **Issue 4: Database Connection Issues**
**Solution**: Check PostgreSQL connection in `.env` file

## 🔄 **Rollback Plan**

If something goes wrong:
1. **Immediate**: Switch back to Supabase AuthProvider
2. **Short-term**: Revert the last component change
3. **Long-term**: Keep Supabase code until Laravel is fully tested

## 📞 **Need Help?**

1. **Check Laravel logs**: `storage/logs/laravel.log`
2. **Verify API responses**: Use browser dev tools
3. **Test endpoints**: Use Postman/Insomnia
4. **Check database**: Verify PostgreSQL connection

## 🎉 **Success Checklist**

- [ ] Authentication working (login/register/logout)
- [ ] Dashboard data loading
- [ ] CRUD operations working
- [ ] File uploads working
- [ ] PDF generation working
- [ ] All pages functional
- [ ] No console errors
- [ ] Performance acceptable

---

**Remember**: Take it slow, test thoroughly, and don't hesitate to rollback if needed. Your existing Supabase code is your safety net! 🛡️
